/*! \file peanotest.c

\brief Test driver for Peano arithmetic. It uses the CUnit framework 

*/

/*! Include files for using CUnit
 */

#include<CUnit/CUnit.h>
#include<CUnit/TestDB.h>
#include <CUnit/Curses.h>

/*! Include the interface to Peano implementation
 */
#include"peano.h"



/*! initialization and cleanup functions
 */

int init(void) { return 0;}
int clean(void) { return 0;}


/*! Self consistency family of tests

The first group of tests checks Peano for consistency. It consists of
tests of 
 \li consistency of equality and object constructors
 \li consistency of different implementations of logics
*/

void 
self_consistent_eq(void)
{
  ASSERT(Zeq(Z()));
  ASSERT(eq(Z(),Z()));
  ASSERT(eq(S(S(Z())),S(S(Z()))));
  ASSERT(!eq(S(Z()),Z()));
  ASSERT(!eq(S(Z()),S(S(Z()))));
}

void 
self_consistent_sum(void)
{
  ASSERT(eq(itsum(Z(),Z()),Z()));
  ASSERT(eq(itsum(S(S(Z())),S(S(Z()))),recsum(S(S(Z())),S(S(Z())))));
}

/*! Model theoretic consistency

The second family is inspired (and never gets to it) on model theoretic 
proofs of faithfulness. We choose the nonnegative integers as a model and
try to prove that the model satisfies our Peano formalization. Everything
under the limitation of a framework (C language) that must be computable.)
*/

void
model_consistent_sum(void)
{
  ASSERT(nn2uns(itsum(Z(),Z())) == 0);
  ASSERT(nn2uns(itsum(S(S(Z())),S(S(Z())))) == 
         nn2uns(recsum(S(S(Z())),S(S(Z())))));

}


/*! Starting up the tests
*/
void runTests(void)
{
  PTestGroup pGroup;
  PTestCase pTest;


  initialize_registry();
  pGroup = add_test_group("Self Consistency", init, clean);
  pTest = add_test_case(pGroup, "Self Test equality", self_consistent_eq);
  pTest = add_test_case(pGroup, "Self Test sum", self_consistent_sum);

  pGroup = add_test_group("Model Consistency", init, clean);
  pTest = add_test_case(pGroup, "Model Test sum", model_consistent_sum);

  curses_run_tests();
  cleanup_registry();
}


int 
main()
{
  
  runTests();
 
  return 0;
}
