/*! \file peano.h
  \brief An interface (header file) for Peano Arithmetic

It defines the representation for Peano numbers and the interface to operations
on them.

*/

/*! \ingroup Peano_arithmetic

*/


/*! \struct Natnum 
\brief A structure for representing Natural Numbers from the Peano Arithmetic
 
    Zero is represented as null pointer to pointer to Natnum
    A number \f$n>0\f$ is represented as a linked sequence of Natnum
*/

/*! \typedef Natnum_t 
\brief Natnum_t : the object that will be operated on

A Peano number will always be a pointer to a Natnum, i.e., a Natnum_t 

*/
typedef struct Natnum *Natnum_t;

struct Natnum{
  Natnum_t s;
};


/*! \fn Z
A constructor for an object representing the number zero

*/

extern Natnum_t Z();

/*! A constructor for an object representing the successor of a given number
\param n a number
\return the successor of n
*/

extern Natnum_t S(const Natnum_t n);

/*! tail recursive function computing the sum of two numbers

Given the numbers, it returns an object representing their sum
*/

extern Natnum_t itsum(const Natnum_t, const Natnum_t);

/*!  recursive function computing the sum of two numbers

Given the numbers, it returns an object representing their sum
*/

extern Natnum_t recsum(const Natnum_t, const Natnum_t);

/*!  iterative function computing the sum of two numbers

Given the numbers, it returns an object representing their sum
*/
extern Natnum_t loopsum(const Natnum_t, const Natnum_t);

/*!  the predecessor of a given number
 */
extern const Natnum_t pred(const Natnum_t);

/*!  whether a number is zero
 */
extern int Zeq(Natnum_t );


/*!  whether two numbers are equal 
 */
extern int eq(Natnum_t, Natnum_t );

/*! Converting a Natnum_t to binary representation
*/
extern unsigned nn2uns(Natnum_t );

/*! From binary to Natnum_t
 */
extern Natnum_t uns2nn(unsigned );





