/*! \file peano.c
\brief The implementation for Peano Arithmetic

Operations on Natnum_t  objects are defined.
 
struct Natnum is a cell containing just a pointer. A linked list with \f$N\f$ of
those cells, followed by a null pointer represents the integer \f$N\f$: \f$N\f$
iterations of \f$S\f$, the sucessor function applied to \f$Z\f$, the zero
constructor.

*/

/*! \ingroup Peano_Implementation 

*/

#include"peano.h"

/*!  a local error procedure

A procedure for producing an error message and aborting the program
 \param s pointer to error message
 \param n an interger constant to be returned to the operating system

*/
static void
error(const char *s, const int n)
{
  printf("%s\n",s);
  exit(n);
}

/*!  An allocator for Natnum

It uses malloc to allocate space for Natnum, possibly aborting the program
\return pointer to newly allocated Natnum
*/

static Natnum_t
Natnum_malloc()
{
  Natnum_t n = (Natnum_t) malloc(sizeof(struct Natnum));
  if (!n) // n is null pointer
    error("Natnum allocation",1);
  return n;
}


Natnum_t 
Z()
{
  return 0; // null pointer representing Z
}

Natnum_t
S(const Natnum_t n)
{
  Natnum_t s = Natnum_malloc();
  s->s = n;
  return s;
}

/*! recursive function for computing sum over Natnum

it computes sum of two Natnum objects, using recursion with suspended computation (computation with draft on stack)

*/
    
Natnum_t 
recsum(const Natnum_t s0, const Natnum_t s1)
{
  Natnum_t s;

  if (Zeq(s1)) 
    return s0;
  return S(recsum(s0,pred(s1)));
}

/*! tail recursive function for computing sum over Natnum

*/
    
Natnum_t
itsum(const Natnum_t s0, const Natnum_t s1)
{
  Natnum_t s;

  if (Zeq(s1)) 
    return s0;
  return itsum(S(s0),pred(s1));
}

/*! The sum of two Natnum object, using iteration

It implements  invariant assertion.
*/

Natnum_t 
loopsum(Natnum_t s0, Natnum_t s1)
{
  Natnum_t count = s1;
  Natnum_t sum = s0;

  while (!Zeq(count)){
#ifdef  _DEBUG_
    assert(nn2uns(sum) == (nn2uns(s0)+(nn2uns(s1)-nn2uns(count))));
#endif
    sum = S(sum);
    count = pred(count);
  }
  return sum;
}


int 
Zeq(Natnum_t n)
{
  return n == 0;
}





const Natnum_t 
pred(const Natnum_t n)
{
  return n->s;
}

/*! equlity test

Notice that equality is defined with respect to the intended representation
*/

int
eq(Natnum_t n0, Natnum_t n1)
{
  if (Zeq(n0)){
    return Zeq(n1);
  }
  else{
    if (Zeq(n1))
      return 0;
    else return eq(pred(n0),pred(n1));
  }
}



unsigned
nn2uns(Natnum_t nn)
{
  unsigned u;

  u = 0;
  while (!Zeq(nn)){
    ++u;
    nn = pred(nn);
  }
  return u;
}


Natnum_t
uns2nn(unsigned u)
{
  Natnum_t nn;

  nn = Z();
  while (u!=0){
    --u;
    nn = S(nn);
  }
  return nn;
}





